-- For debugging purposes
--
function table.val_to_str ( v )
  if "string" == type( v ) then
    v = string.gsub( v, "\n", "\\n" )
    if string.match( string.gsub(v,"[^'\"]",""), '^"+$' ) then
      return "'" .. v .. "'"
    end
    return '"' .. string.gsub(v,'"', '\\"' ) .. '"'
  else
    return "table" == type( v ) and table.tostring( v ) or
      tostring( v )
  end
end

function table.key_to_str ( k )
  if "string" == type( k ) and string.match( k, "^[_%a][_%a%d]*$" ) then
    return k
  else
    return "[" .. table.val_to_str( k ) .. "]"
  end
end

function table.tostring( tbl )
  local result, done = {}, {}
  for k, v in ipairs( tbl ) do
    table.insert( result, table.val_to_str( v ) )
    done[ k ] = true
  end
  for k, v in pairs( tbl ) do
    if not done[ k ] then
      table.insert( result,
        table.key_to_str( k ) .. "=" .. table.val_to_str( v ) )
    end
  end
  return "{" .. table.concat( result, "," ) .. "}"
end
--]]

--[[ storage table data structure
storage.
  chargingTrains
  ... trainID as key
    {train entity, list of {charging locomotive entity, dummy charging entity, max fuel value for locomotive}}

  locomotiveList
  ...
    {locomotive entity name, dummy fuel item, energy capacity}
--]]

-- data = {locomotive entity name, dummy fuel item name}
function registerLocomotive(data)
  local newLocomotive = data
  if prototypes.entity[data[1]].valid then
    if data[2] ~= nil and prototypes.item[data[2]].valid then
      newLocomotive[3] = prototypes.item[data[2]].fuel_value
    else
      newLocomotive[2] = "No storage"
      newLocomotive[3] = 0
    end
    if storage.locomotiveList == nil then storage.locomotiveList = {} end
    table.insert(storage.locomotiveList, newLocomotive)
  end
end

--
remote.add_interface("electrictrains", {
  status = function()
    local list
    local entry
    for i,v in pairs(storage.chargingTrains) do
      list = "Train " .. i .. ":"
      for j,w in pairs(v[2]) do
        entry = "\nLocomotive " .. w[1].unit_number .. " '" .. w[1].backer_name .. "':\n\tType: " .. w[1].name .. "\n\tCurrent fuel: " .. w[1].burner.currently_burning.name.name .. "\nFuel value: " .. w[1].burner.remaining_burning_fuel .. " / " .. w[1].burner.currently_burning.name.fuel_value
        list = list .. entry
      end
      game.print(list)
    end
    --game.print("Charging trains: " .. table.tostring(storage.chargingTrains))
    game.print("EEIs: " .. table.tostring(game.surfaces[1].find_entities_filtered{name = "deg-locomotive-charging-dummy"}))
    game.print("Locomotive dictionary: " .. table.tostring(storage.locomotiveList))
  end,

  register = function(entity_name, dummy_fuel_name)
    registerLocomotive({entity_name, dummy_fuel_name})
  end
})


local waitStation = defines.train_state.wait_station


script.on_init(function()
  storage.chargingTrains = {}
  storage.locomotiveList = {}
  registerLocomotive({"deg-electric-locomotive", "deg-electric-locomotive-fuel-dummy"})
end)

script.on_load(function()
  if next(storage.chargingTrains) ~= nil then
    script.on_nth_tick(23, tick)
  end
end)

script.on_configuration_changed(function()
  storage.locomotiveList = {}
  registerLocomotive({"deg-electric-locomotive", "deg-electric-locomotive-fuel-dummy"})
end)


function locomotiveMatch(entity_name)
  for i,v in pairs(storage.locomotiveList) do
    if v[1] == entity_name then
      return i
    end
  end

  return 0
end

function fuelMatch(fuel_name)
  for i,v in pairs(storage.locomotiveList) do
    if v[2] == fuel_name then
      return i
    end
  end

  return 0
end

-- Sets a new electric locomotive's internal fuel item and gives it a small amount of energy to reach a station and start charging.
function entity_built(event)
  local i = locomotiveMatch(event.entity.name)
  if i > 0 and storage.locomotiveList[i][3] ~= 0 then
    event.entity.burner.currently_burning = {name = storage.locomotiveList[i][2], quality = event.entity.quality}
    event.entity.burner.remaining_burning_fuel = 100000
    --game.print("New train charge: " .. event.entity.burner.remaining_burning_fuel)
  end
end
script.on_event(defines.events.on_built_entity, entity_built, {{filter = "rolling-stock"}})
script.on_event(defines.events.on_robot_built_entity, entity_built, {{filter = "rolling-stock"}})


function entity_mined(event)
  local i = locomotiveMatch(event.entity.name)
  if i > 0 and event.entity.burner.currently_burning ~= nil and event.entity.burner.currently_burning.name.name ~= storage.locomotiveList[i][2] and event.entity.burner.currently_burning.name.burnt_result ~= nil then
    event.buffer.insert({name=event.entity.burner.currently_burning.name.burnt_result.name, quality = event.entity.burner.currently_burning.quality})
  end
end
script.on_event(defines.events.on_player_mined_entity, entity_mined, {{filter = "rolling-stock"}})
script.on_event(defines.events.on_robot_mined_entity, entity_mined, {{filter = "rolling-stock"}})


function train_changed_state(event)
  if event.train.state == waitStation then
    local locomotives = {}
    for i,v in pairs(event.train.locomotives) do -- for the list of locomotives...
      for j,w in pairs(v) do -- for the forward/backward sublists of locomotives in that list...
        for k,x in pairs(storage.locomotiveList) do -- if it shows up in the registered list of chargeable locomotives...
          if x[1] == w.name and x[3] > 0 and x[3] > w.burner.remaining_burning_fuel then -- AND has internal max storage greater than whatever it's using now, add to the charging list
            if w.burner.currently_burning ~= nil and w.burner.currently_burning.name.name ~= x[2] then
              if w.burner.currently_burning.name.burnt_result ~= nil then
                w.burner.burnt_result_inventory.insert({name = w.burner.currently_burning.name.burnt_result.name, quality = w.burner.currently_burning.quality})
              end
            end
            w.burner.currently_burning = {name = x[2], quality = w.quality}
            local charger = game.surfaces[w.surface.index].create_entity{name = "deg-locomotive-charging-dummy", position = w.position, force = w.force}
            charger.electric_buffer_size = x[3]
            charger.energy = w.burner.remaining_burning_fuel
            charger.destructible = false
            table.insert(locomotives, {w,charger,x[3]})
            break
          end
        end
      end
    end

    if next(locomotives) ~= nil then
      storage.chargingTrains[event.train.id] = {event.train, locomotives}
      --[[
      --game.print("Train " .. event.train.id .. " charging.")
      for i,v in pairs(locomotives) do
        game.print("Locomotive " .. v[1].unit_number .. " charging " .. v[1].burner.currently_burning.name .. " via EEI " .. v[2].unit_number .. ", from " .. v[1].burner.remaining_burning_fuel .. " to " .. v[3] .. " fuel value.")
      end --]]
      activateTicker()
    end
  elseif event.old_state == waitStation and storage.chargingTrains[event.train.id] ~= nil then
    for i,v in pairs(storage.chargingTrains[event.train.id][2]) do
        v[2].destroy()
    end
  end
end
script.on_event(defines.events.on_train_changed_state, train_changed_state)


function activateTicker()
  if next(storage.chargingTrains) ~= nil then
    script.on_nth_tick(23, tick)
  end
end


function tick()
  for i,v in pairs(storage.chargingTrains) do
    if not v[1].valid or v[1].state ~= waitStation or #v[2] == 0 then
      for j,w in pairs(v[2]) do
        w[2].destroy()
      end
      storage.chargingTrains[i] = nil
    else
      for j,w in pairs(v[2]) do
        w[1].burner.remaining_burning_fuel = w[2].energy
        --game.print("Train charge: " .. w[2].energy .. " : " .. w[1].burner.remaining_burning_fuel)
        if w[1].burner.remaining_burning_fuel == w[3] then
          w[2].destroy()
          table.remove(v[2],j)
          --game.print("Locomotive " .. w[1].unit_number .. " finished charging.")
        end
      end
    end
  end
  
  if next(storage.chargingTrains) == nil then
    script.on_nth_tick(nil)
    --game.print("No locomotives left to charge, deactivating ticker.")
  end
end
